package ga.core.evaluation;

import ga.core.algorithm.interactive.ISIGA;
import ga.core.individual.IIndividual;

/**
 * Interface for interactive fitness evaluators.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public interface IInteractiveFitnessEvaluator<T extends IIndividual<T>> extends
    IFitnessEvaluator<T> {
  @Override
  public void evaluate(T individual);

  /**
   * Adds an {@link EvaluationListener} that will be notified about the
   * evaluation state.
   * 
   * @param listener
   *          The listener.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void addEvaluationListener(EvaluationListener<T> listener);

  /**
   * Adds an {@link EvaluationListener}.
   * 
   * @param listener
   *          The listener.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void removeEvaluationListener(EvaluationListener<T> listener);

  /**
   * Notification method that causes all {@link EvaluationListener}s being
   * notified that the evaluator demands a new individual for evaluation.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void fireNewIndividualRequested();

  /**
   * Notification method that causes all {@link EvaluationListener}s being
   * notified that the evaluator has evaluated the individual.
   * 
   * @param individual
   *          The evaluated individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void fireIndividualEvaluated(T individual);

  /**
   * Sets the interactive genetic algorithm.
   * 
   * @param algorithm
   *          The SIGA.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setAlgorithm(ISIGA<T> algorithm);

  /**
   * Getter for the interactive genetic algorithm.
   * 
   * @return The SIGA.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public ISIGA<T> getAlgorithm();
}
